#!/usr/bin/python
# coding: utf-8

'''
eartraining.py, Dec. 11, 2009

This is a quick-and-dirty implementation of some ideas about ear training.
See http://www.matthieuamiguet.ch/actus/eartraining for more details.
Warning: this is only a proof-of-concept implementation. It would need *much*
more work before it could be useful as a pedagogical tool!

To make it work, you will need:
- Python: www.python.org
- MMA: http://www.mellowood.ca/mma
- A midi file player - e.g. timidity if you use Linux

It's also using easygui, but it's included in the download to make things a little bit easier.

To make it work:
- Download and install all the dependencies above
- If you don't use timidity, modify the line below that reads
midi_player = "timidity"
and replace timidity with whatever midi player you're using.
- Launch this script

For feedback, questions or ideas, please contact me at
http://matthieuamiguet.ch/contact

Matthieu Amiguet, 2009
'''

import platform
import sys
import os
from signal import SIGINT
import random
import subprocess
from easygui import choicebox, msgbox

nb_repeats = 20

# ----- Setting the midi player

if platform.system() == 'Windows':
	midi_player = "playsmf"
	mma = r"c:\mma\mma"
else:
	midi_player = "timidity"
	mma = "mma"

# ----- killing the midi process -----

if platform.system() == 'Windows':
	def kill(pid, dummy):
		"""the poor man's kill...
		It's a bit of a hack, but it should work on Windows XP and later...
		"""
		subprocess.Popen("taskkill /F /T /PID %i"%pid , shell=True)
else:
	try:
		from os import kill
	except ImportError:
		msgbox("Can't find out how to kill processes on your platform. Aborting.")
		sys.exit(1)

# ----- reading files -----------
def get_grooves(filename="grooves.txt"):
	with open(filename) as f:
		return [l.strip() for l in f if not l.startswith('#')]

grooves = get_grooves()

def get_patterns(filename="patterns.txt"):
	d = {}
	
	with open(filename) as f:
		for l in f:
			l = l.strip()
			if l.startswith('#'):
				continue
			name, pattern = l.split(':')
			pattern = pattern.strip()
			
			key = "%s (%s)" % (name, pattern)
			value = [m.strip() for m in pattern.split("|")]
			
			d[key] = value
			
	return d

		
# ----- transposition -----------

notes = [
	'C', 'C#','D', 'D#', 'E', 'F', 'F#', 'G', 'G#', 'A', 'A#', 'B'
]

naturals2pos = {
	'C': 0,
	'D': 2,
	'E': 4,
	'F': 5,
	'G': 7,
	'A': 9,
	'B': 11
}

def transpose(pattern, amount=None):
	''' Quick-and dirty way of transposing patterns.
	
	amount is in semitones, must be between 0 and 11.
	if not given, will be selected randomly.
	'''
	
	if not amount:
		amount = random.randint(0,11)
	
	transposed = []

	for chord in pattern:
		root = naturals2pos[chord[0]]
		if chord[1] == "#":
			root = (root+1) % 12
			nature = chord[2:]
		elif chord[1] == "b":
			root = (root-1) % 12
			nature = chord[2:]
		else:
			nature = chord[1:]
		
		newroot = (root + amount)  % 12
		transposed.append("%s%s" % (notes[newroot], nature))
		
	return transposed

# ------- MMA stuff -------

# template for MMA's input files
mma_template = '''
Tempo %d
Groove %s
%s
'''

def mma_from_pattern(pattern, filename='to_play.mma', amount_transpose=None):
	groove = random.choice(grooves)
	trans = transpose(pattern, amount_transpose)*nb_repeats
	tempo = random.randint(10,18)*10
	to_play = mma_template % (tempo, groove,'\n'.join(trans))
	with open(filename, 'w') as f:
		f.write(to_play)
		
def mma2mid(infile='to_play.mma'):
	# subprocess.call doesn't seem to work here on Windows
	# it works in play_midi, though... go figure!
	problem = os.system("%s %s" %(mma, infile))
	if problem:
		msgbox("Problem running MMA. Exiting")
		sys.exit(1)
	
def play_mid(filename='to_play.mid'):
	return subprocess.Popen([midi_player,filename], stdout=subprocess.PIPE, stderr=subprocess.PIPE).pid
	
def play(pattern):
	mma_from_pattern(pattern)
	mma2mid()
	return play_mid()
	
if __name__ == '__main__':
	patterns = get_patterns()
	
	remaining = patterns.keys()
	
	while remaining:
		
		name = random.choice(remaining)
		pid = play(patterns[name])
		
		message = "Name that pattern!"
		answer = None
		while answer != name:
			answer = choicebox(message,'Question',remaining, buttons=['OK', 'Quit'])
			if not answer:
				kill(pid, SIGINT)
				sys.exit(0)
			message = "That's wrong. Try again!"
		
		remaining.remove(name)
		kill(pid, SIGINT)

